<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Server;
use App\Models\StoreAction;
use App\Models\Transaction;
use App\Helpers\SteamHelper;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class ServerActionController extends Controller
{
    /**
     * Debug endpoint to check what's in the database
     */
    public function debug(Request $request)
    {
        $server = $request->attributes->get('server');

        // Get ALL store actions (not just unexecuted)
        $allActions = StoreAction::with(['user', 'package', 'transaction'])
            ->orderBy('created_at', 'desc')
            ->take(10)
            ->get();

        // Get ALL transactions
        $allTransactions = Transaction::with(['user', 'package', 'server'])
            ->where('status', 'completed')
            ->orderBy('created_at', 'desc')
            ->take(10)
            ->get();

        // Get unexecuted actions for this server specifically (including null server_id)
        $pendingForServer = StoreAction::unexecuted()
            ->where(function($query) use ($server) {
                $query->where('server_id', $server->id)
                      ->orWhereNull('server_id');
            })
            ->with(['user', 'package', 'transaction'])
            ->get();

        // Get all unexecuted actions regardless of server
        $allPending = StoreAction::unexecuted()
            ->with(['user', 'package', 'transaction'])
            ->get();

        return response()->json([
            'success' => true,
            'debug_info' => [
                'server_id' => $server->id,
                'server_name' => $server->name,
                'total_store_actions' => StoreAction::count(),
                'total_completed_transactions' => Transaction::where('status', 'completed')->count(),
                'pending_actions_for_server' => $pendingForServer->count(),
                'all_pending_actions' => $allPending->count(),
                'recent_transactions' => $allTransactions->map(function($t) {
                    return [
                        'id' => $t->id,
                        'user' => $t->user ? $t->user->name : 'No User',
                        'package' => $t->package ? $t->package->name : 'No Package',
                        'server' => $t->server ? $t->server->name : 'No Server',
                        'status' => $t->status,
                        'created_at' => $t->created_at,
                    ];
                }),
                'recent_actions' => $allActions->map(function($a) {
                    return [
                        'id' => $a->id,
                        'user' => $a->user ? $a->user->name : 'No User',
                        'package' => $a->package ? $a->package->name : 'No Package',
                        'server_id' => $a->server_id,
                        'executed' => $a->executed,
                        'actions' => $a->actions,
                        'created_at' => $a->created_at,
                    ];
                }),
                'pending_for_server' => $pendingForServer->map(function($a) {
                    return [
                        'id' => $a->id,
                        'user' => $a->user ? $a->user->name : 'No User',
                        'user_steamid' => $a->user ? $a->user->steamid : 'No SteamID',
                        'package' => $a->package ? $a->package->name : 'No Package',
                        'actions' => $a->actions,
                        'created_at' => $a->created_at,
                    ];
                })
            ],
            'timestamp' => Carbon::now()->toISOString()
        ]);
    }

    /**
     * Get pending actions for a server.
     * Called by Gmod server every 60 seconds.
     */
    public function getPendingActions(Request $request)
    {
        $server = $request->attributes->get('server');

        // Get unexecuted actions for this server (including actions with no server assignment)
        $pendingActions = StoreAction::unexecuted()
            ->where(function($query) use ($server) {
                $query->where('server_id', $server->id)
                      ->orWhereNull('server_id'); // Include actions with no server assignment
            })
            ->with(['user', 'package', 'transaction'])
            ->orderBy('created_at', 'asc')
            ->get();

        // Format actions for Gmod server
        $formattedActions = $pendingActions->map(function ($action) {
            // Add null checks
            $user = $action->user;
            $package = $action->package;
            $transaction = $action->transaction;
            
            return [
                'id' => $action->id,
                'user_id' => $action->user_id,
                'user_steamid' => $user ? $user->steamid : null,
                'user_name' => $user ? $user->name : 'Unknown User',
                'package_name' => $package ? $package->name : 'Unknown Package',
                'transaction_id' => $action->transaction_id,
                'actions' => $this->formatActionsForGmod($action->actions ?? [], $user, $package, $transaction),
                'created_at' => $action->created_at->toISOString(),
            ];
        });

       // Log::info("Server {$server->name} polled for actions. Found " . $pendingActions->count() . " pending actions.");

        return response()->json([
            'success' => true,
            'server_id' => $server->id,
            'server_name' => $server->name,
            'details' => $formattedActions,
            'count' => $pendingActions->count(),
            'timestamp' => Carbon::now()->toISOString()
        ]);
    }

    /**
     * Format actions for Gmod consumption
     */
    private function formatActionsForGmod($actions, $user = null, $package = null, $transaction = null)
    {
        if (!is_array($actions)) {
            return [];
        }
        
        $formattedActions = [];
        
        foreach ($actions as $action) {
            if (!is_array($action) || !isset($action['type'])) {
                continue;
            }
            
            $formattedAction = [
                'type' => $action['type']
            ];
            
            switch ($action['type']) {
                case 'rank':
                    $formattedAction['rank'] = $action['rank'] ?? 'vip';
                    $formattedAction['duration'] = $action['duration'] ?? 0; // 0 = permanent
                    break;
                    
                case 'money':
                    $formattedAction['amount'] = floatval($action['amount'] ?? 0);
                    break;
                    
                case 'custom_lua':
                    $formattedAction['lua_code'] = $this->replaceStringPlaceholders($action['lua_code'] ?? '', $user, $package, $transaction);
                    break;
                    
                case 'console_command':
                    $formattedAction['command'] = $this->replaceStringPlaceholders($action['command'] ?? '', $user, $package, $transaction);
                    break;
                    
                default:
                    // Pass through any other action data
                    $formattedAction = array_merge($formattedAction, $action);
                    break;
            }
            
            $formattedActions[] = $formattedAction;
        }
        
        return $formattedActions;
    }

    /**
     * Replace string placeholders with actual values
     */
    private function replaceStringPlaceholders($string, $user = null, $package = null, $transaction = null)
    {
        if (!is_string($string)) {
            return $string;
        }

        $replacements = [
            // User placeholders
            '{user_name}' => $user ? $user->name : 'Unknown User',
            '{user_steamid64}' => $user ? $user->steamid : 'Unknown SteamID',
            '{user_steamid}' => $user ? SteamHelper::steamId64ToLegacy($user->steamid) : 'Unknown SteamID',
            '{user_id}' => $user ? $user->id : 0,
            
            // Package placeholders
            '{package_name}' => $package ? $package->name : 'Unknown Package',
            '{package_description}' => $package ? $package->description : 'Unknown Description',
            '{package_type}' => $package ? $package->type : 'unknown',
            '{package_slug}' => $package ? $package->slug : 'unknown',
            
            // Transaction placeholders
            '{package_price}' => $transaction ? $transaction->amount : ($package ? $package->price : 0),
            '{original_price}' => $transaction ? $transaction->original_amount : ($package ? $package->price : 0),
            '{discount_amount}' => $transaction ? $transaction->discount_amount : 0,
            '{transaction_id}' => $transaction ? $transaction->id : 0,
            '{payment_method}' => $transaction ? $transaction->payment_method : 'unknown',
            
            // Date placeholders
            '{current_date}' => date('Y-m-d'),
            '{current_time}' => date('H:i:s'),
            '{current_datetime}' => date('Y-m-d H:i:s'),
            '{timestamp}' => time(),
        ];

        return str_replace(array_keys($replacements), array_values($replacements), $string);
    }

    /**
     * Mark an action as completed.
     * Called by Gmod server after successfully executing an action.
     */
    public function markActionComplete(Request $request, $actionId)
    {
        $server = $request->attributes->get('server');

        // Find the action (allow null server_id or matching server_id)
        $action = StoreAction::where('id', $actionId)
            ->where(function($query) use ($server) {
                $query->where('server_id', $server->id)
                      ->orWhereNull('server_id');
            })
            ->first();

        if (!$action) {
            return response()->json([
                'success' => false,
                'error' => 'Action not found or not assigned to this server'
            ], 404);
        }

        if ($action->executed) {
            return response()->json([
                'success' => false,
                'error' => 'Action already executed'
            ], 400);
        }

        // Mark as executed
        $action->update([
            'executed' => true,
            'received_at' => Carbon::now(),
        ]);

        Log::info("Server {$server->name} marked action {$actionId} as completed for user {$action->user->name} (SteamID: {$action->user->steamid})");

        return response()->json([
            'success' => true,
            'message' => 'Action marked as completed',
            'action_id' => $actionId,
            'timestamp' => Carbon::now()->toISOString()
        ]);
    }

    /**
     * Mark multiple actions as completed.
     * Called by Gmod server after successfully executing multiple actions.
     */
    public function markActionsComplete(Request $request)
    {
        $server = $request->attributes->get('server');
        $actionIds = $request->input('action_ids', []);
        
        if (empty($actionIds) || !is_array($actionIds)) {
            return response()->json([
                'success' => false,
                'error' => 'Valid action_ids array is required'
            ], 400);
        }

        // Find and update actions (allow null server_id or matching server_id)
        $updated = StoreAction::whereIn('id', $actionIds)
            ->where(function($query) use ($server) {
                $query->where('server_id', $server->id)
                      ->orWhereNull('server_id');
            })
            ->where('executed', false)
            ->update([
                'executed' => true,
                'received_at' => Carbon::now(),
            ]);

        Log::info("Server {$server->name} marked {$updated} actions as completed");

        return response()->json([
            'success' => true,
            'message' => "Marked {$updated} actions as completed",
            'updated_count' => $updated,
            'timestamp' => Carbon::now()->toISOString()
        ]);
    }

    /**
     * Get server status (optional heartbeat endpoint).
     */
    public function serverHeartbeat(Request $request)
    {
        $server = $request->attributes->get('server');

        // You can update last_seen timestamp if you add that column to servers table
        // $server->update(['last_seen' => Carbon::now()]);

        $pendingCount = StoreAction::unexecuted()
            ->where(function($query) use ($server) {
                $query->where('server_id', $server->id)
                      ->orWhereNull('server_id');
            })
            ->count();

        return response()->json([
            'success' => true,
            'server_id' => $server->id,
            'server_name' => $server->name,
            'pending_actions' => $pendingCount,
            'timestamp' => Carbon::now()->toISOString()
        ]);
    }

    /**
     * Test endpoint to verify API key (optional).
     */
    public function testConnection(Request $request)
    {
        $server = $request->attributes->get('server');

        return response()->json([
            'success' => true,
            'message' => 'Connection successful',
            'server_id' => $server->id,
            'server_name' => $server->name,
            'timestamp' => Carbon::now()->toISOString()
        ]);
    }

    /**
     * Create a test action for debugging purposes
     */
    public function createTestAction(Request $request)
    {
        $server = $request->attributes->get('server');

        // Create a test store action
        $testAction = StoreAction::create([
            'user_id' => 1, // Assuming user ID 1 exists
            'server_id' => $server->id,
            'transaction_id' => null, // No transaction for test
            'package_id' => 1, // Assuming package ID 1 exists
            'actions' => [
                [
                    'type' => 'rank',
                    'rank' => 'vip',
                    'duration' => 0
                ],
                [
                    'type' => 'money',
                    'amount' => 1000
                ]
            ],
            'executed' => false,
            'received_at' => null,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Test action created',
            'test_action_id' => $testAction->id,
            'timestamp' => Carbon::now()->toISOString()
        ]);
    }
}